
(* this module is a set of utilities to deal with the filesystem
   in particular, reading files, recursive rmdir, mkdir, etc. 
   
   dependencies: Buffer module *)
module File =
    
(* returns the contents of a file as a string
   if an error occurs, returns an empty string 
   this version can be used to read something that's not a real file (that has no file size of example) *)
let readfile filename = 
  (* knowing the file size would allow us the allocate a buffer of the right size *)
  (*let filesize = 
    case Posix.stat filename 
     | Posix.Result st -> st.Posix.st_size
     | Posix.PosixError _ -> 0
    end in *)
    
  let s = String.build 4096 in 
  let buf = Buffer.build 4096 in 

  let rec readfiletobuffer fd = 
     let len = Posix.read fd s 4096 0 in 
     if len > 0 then 
       let () = Buffer.add_substring buf s 0 len in 
       readfiletobuffer fd
     else if len = 0 then (* len = 0 indicates we have reached the end of the file *)
       ()
     else (* len < 0 indicates there is an error, we return an empty buffer *)
       Buffer.clear buf
   in 
   
  let fd = Posix.open filename [| Posix.O_RDONLY |] in 
  let res = readfiletobuffer fd in 
  let _ = Posix.close fd in 
  
  Buffer.contents buf
  
  
(* this writes a string into a file
   the string can be big 
   returns true if successful, false if not *)
let write_string_to_file filename s = 

  let rec writestringtofile_aux fd pos remaining_size = 
     if remaining_size <= 0 then 
       true
     else 
       let len = Posix.write fd s (min 8192 remaining_size) pos in (* we write by chunks of 8k max *)
       if len > 0 then 
         writestringtofile_aux fd (pos + len) (remaining_size - len)
       else
         false
    in 
    
  let fd = Posix.open filename [| Posix.O_CREAT; Posix.O_TRUNC; Posix.O_WRONLY |] in 
  let result = writestringtofile_aux fd 0 (String.length s) in 
  let _ = Posix.close fd in 
  
  result



(* same as mkdir -p , but expects an absolute path, and will do nothing if not 
   TODO: this is not the same semantics as mkdir, so it might be a problem. 
         => check and provide a reference to POSIX *)
let mkdir path = 
  let split_path = String.split path '/' in 
  case split_path
  | [] -> () (* incorrect absolute path *)
  | "" :: components -> (* correct absolute path *)
     let _ = List.fold (fun absolute_path dir_name -> let full_dir = absolute_path ^ "/" ^ dir_name in 
                                                      let _ = Posix.mkdir full_dir 0o700 in 
                                                      full_dir)  ""  components in 
     ()
  | _ :: _ -> () (* incorrect absolute path *)
  end
  
  
(* if dir=/foo/bar/ , this function removes what inside /foo/bar/ , but not /foo/bar/ itself *)
let remove_r dir =

  (* reads directory contents and filters out "." and ".." *)
  let get_entries path = 
    let fd = Posix.open path [| Posix.O_RDONLY |] in
    let (ret, entries) = Posix.readdir fd in
    let _ = Posix.close fd in
    let helper tmp element = 
      case element 
      | "."  -> tmp
      | ".." -> tmp
      | _ -> element :: tmp
      end in
    Array.fold helper [] entries in (* returns the filtered results *) 
    
  (* true iff path is dir *)
  let is_dir path = 
    case Posix.lstat path 
    | Posix.PosixError _ -> false
    | Posix.Result stat  -> stat.Posix.st_mode land 0xF000 = 0x4000
    end in
  
  let get_parent_directory path = 
    case String.findchar_fromright path '/' (String.length path - 1)
    | None -> "" (* cannot happen *)
    | Some p -> String.sub path 0 p
    end in
  
  (* tail-recursive, because there can be lots of entries *)
  let rec walker dirpath = 
    let full_path_entries = List.map_rev  (fun entry -> dirpath ^ "/" ^ entry)  (get_entries dirpath) in (* tail recursive *)
    let () = List.iter (fun path -> let _ = Posix.unlink path in ()) full_path_entries in (* remove all files - unlink won't remove directories *)
    let directory_entry_opt = List.find (fun entry -> is_dir entry) full_path_entries in (* find the first directory *)
    case directory_entry_opt
    | None -> (* remove dirpath and recurse, unless it's the top level one, in which case we're done *)
      if dirpath = dir then  
        () (* we're done *)
      else
        let _ = Posix.rmdir dirpath in 
        let new_dirpath = get_parent_directory dirpath in 
        walker new_dirpath
    | Some somedir -> walker somedir (* recurse in the first directory found *)
    end in
    
    walker dir
    
    
    
    
endmodule
